/* Copyright (C) 1999 Lucent Technologies */
/* Kod z ksiki The Practice of Programming */
/* Briana W. Kernighana i Roba Pike'a */

/*
 * Generator tekstu z losowych acuchw Markowa.
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "eprintf.h"

enum {
	NPREF	= 2,	/* Liczba sw w przedrostku */ 
	NHASH	= 4093,	/* Rozmiar tablicy mieszania przechowujcej stany */ 
	MAXGEN	= 10000	/* Maksymalna liczba sw, jaka moe zosta wygenerowana */ 
};

typedef struct State State;
typedef struct Suffix Suffix;

struct State {	/* przedrostek + lista przyrostkw */
	char	*pref[NPREF];	/* Sowa przedrostka */
	Suffix	*suf;			/* Lista przyrostkw */
	State	*next;			/* Nastpny na licie przyrostkw */
};

struct Suffix {	/* Lista przyrostkw */
	char	*word;			/* Przyrostek */
	Suffix	*next;			/* Nastpny na licie przyrostkw */
};

State	*lookup(char *prefix[], int create);
void	build(char *prefix[], FILE*);
void	generate(int nwords);
void	add(char *prefix[], char *word);

State	*statetab[NHASH];	/* hash table of states */

char NONWORD[] = "\n";  /* Nie moe wystpi jako zwyke sowo */

/* main: generuje losowe acuchy Markowa */
int main(void)
{
	int i, nwords = MAXGEN;
	char *prefix[NPREF];		/* Biecy przedrostek wejciowy */

	int c;
	long seed;

	setprogname("markov");
	seed = time(NULL);

	srand(seed);
	for (i = 0; i < NPREF; i++)	/* Pocztkowy przedrostek */
		prefix[i] = NONWORD;
	build(prefix, stdin);
	add(prefix, NONWORD);
	generate(nwords);
	return 0;
}   

const int MULTIPLIER = 31;  /* Dla funkcji hash() */

/* hash: oblicza warto mieszania dla tablicy NPREF acuchw */
unsigned int hash(char *s[NPREF])
{
	unsigned int h;
	unsigned char *p;
	int i;

	h = 0;
	for (i = 0; i < NPREF; i++)
		for (p = (unsigned char *) s[i]; *p != '\0'; p++)
			h = MULTIPLIER * h + *p;
	return h % NHASH;
}

/* lookup: szuka przedrostka i w razie potrzeby go tworzy.  */ 
/* Zwraca wskanik na znaleziony lub utworzony przedrostek albo NULL w pozostaych przypadkach. */
/*  Przy tworzeniu przedrostkw nie jest wywoywana funkcja strdup, co oznacza, e acuchw nie wolno pniej zmienia. */ 
State* lookup(char *prefix[NPREF], int create)
{
	int i, h;
	State *sp;

	h = hash(prefix);
	for (sp = statetab[h]; sp != NULL; sp = sp->next) {
		for (i = 0; i < NPREF; i++)
			if (strcmp(prefix[i], sp->pref[i]) != 0)
				break;
		if (i == NPREF)		/* Znaleziono */
			return sp;
	}
	if (create) {
		sp = (State *) emalloc(sizeof(State));
		for (i = 0; i < NPREF; i++)
			sp->pref[i] = prefix[i];
		sp->suf = NULL;
		sp->next = statetab[h];
		statetab[h] = sp;
	}
	return sp;
}

/* addsuffix: dodaje do stanu. Przyrostek nie moe si pniej zmieni */
void addsuffix(State *sp, char *suffix)
{
	Suffix *suf;

	suf = (Suffix *) emalloc(sizeof(Suffix));
	suf->word = suffix;
	suf->next = sp->suf;
	sp->suf = suf;
}

/* add: dodaje sowo do listy przyrostkw i aktualizuje tablic przedrostkw */
void add(char *prefix[NPREF], char *suffix)
{
	State *sp;

	sp = lookup(prefix, 1);  /* Utwrz, jeli nie znajdziesz */
	addsuffix(sp, suffix);
	/* Przesunicie sw w tablicy przedrostkw */
	memmove(prefix, prefix+1, (NPREF-1)*sizeof(prefix[0]));
	prefix[NPREF-1] = suffix;
}

/* build: wczytuje dane i tworzy tablic przedrostkw */
void build(char *prefix[NPREF], FILE *f)
{
	char buf[100], fmt[10];

	/* Tworzy acuch formatowania; %s moe wywoa przepenienie bufora */
	sprintf(fmt, "%%%ds", sizeof(buf)-1);
	while (fscanf(f, fmt, buf) != EOF)
		add(prefix, estrdup(buf));
}

/* generate: tworzy dane wyjciowe, po jednym sowie na wiersz */
void generate(int nwords)
{
	State *sp;
	Suffix *suf;
	char *prefix[NPREF], *w;
	int i, nmatch;

	for (i = 0; i < NPREF; i++)	/* Wyzerowanie pocztkowego prefiksu */
		prefix[i] = NONWORD;

	for (i = 0; i < nwords; i++) {
		sp = lookup(prefix, 0);
		if (sp == NULL)
			eprintf("Bd wewntrzny: operacja przeszukiwania nie powioda si");
		nmatch = 0;
		for (suf = sp->suf; suf != NULL; suf = suf->next)
			if (rand() % ++nmatch == 0) /* Prawdopodobiestwio = 1/nmatch */
				w = suf->word;
		if (nmatch == 0)
			eprintf("Bd wewntrzny: brak przyrostka %d %s", i, prefix[0]);
		if (strcmp(w, NONWORD) == 0)
			break;
		printf("%s\n", w);
		memmove(prefix, prefix+1, (NPREF-1)*sizeof(prefix[0]));
		prefix[NPREF-1] = w;
	}
}
